'use client';

import { useState, useEffect } from 'react';
import { ContextBar, Theme, DeviceType } from './ContextBar';
import { DeviceFrame } from './DeviceFrame';

interface PrototypeShellProps {
  title: string;
  who: string;
  what: string;
  why: string;
  children: React.ReactNode;
}

export function PrototypeShell({
  title,
  who,
  what,
  why,
  children,
}: PrototypeShellProps) {
  const [theme, setTheme] = useState<Theme>('light');
  const [device, setDevice] = useState<DeviceType>('desktop');
  const [mounted, setMounted] = useState(false);

  // Load saved preferences on mount
  useEffect(() => {
    setMounted(true);
    const savedTheme = localStorage.getItem('wireframe-theme') as Theme;
    const savedDevice = localStorage.getItem('wireframe-device') as DeviceType;
    
    if (savedTheme && ['light', 'dark', 'wireframe'].includes(savedTheme)) {
      setTheme(savedTheme);
    }
    if (savedDevice && ['desktop', 'ipad', 'iphone'].includes(savedDevice)) {
      setDevice(savedDevice);
    }
  }, []);

  // Apply theme to document
  useEffect(() => {
    if (mounted) {
      document.documentElement.setAttribute('data-theme', theme);
      localStorage.setItem('wireframe-theme', theme);
    }
  }, [theme, mounted]);

  // Save device preference
  useEffect(() => {
    if (mounted) {
      localStorage.setItem('wireframe-device', device);
    }
  }, [device, mounted]);

  const handleThemeChange = (newTheme: Theme) => {
    setTheme(newTheme);
  };

  const handleDeviceChange = (newDevice: DeviceType) => {
    setDevice(newDevice);
  };

  // Prevent flash of wrong theme
  if (!mounted) {
    return null;
  }

  return (
    <>
      <ContextBar
        title={title}
        who={who}
        what={what}
        why={why}
        currentTheme={theme}
        currentDevice={device}
        onThemeChange={handleThemeChange}
        onDeviceChange={handleDeviceChange}
      />
      <DeviceFrame device={device}>
        {children}
      </DeviceFrame>
    </>
  );
}
