'use client';

import { useEffect, useRef, useState } from 'react';

export type DeviceType = 'desktop' | 'ipad' | 'iphone';

interface DeviceConfig {
  width: number;
  height: number;
  name: string;
}

const DEVICE_CONFIGS: Record<DeviceType, DeviceConfig> = {
  desktop: { width: 0, height: 0, name: 'Desktop' }, // 0 means full width
  ipad: { width: 1024, height: 768, name: 'iPad' },
  iphone: { width: 390, height: 844, name: 'iPhone 14' },
};

interface DeviceFrameProps {
  device: DeviceType;
  children: React.ReactNode;
}

export function DeviceFrame({ device, children }: DeviceFrameProps) {
  const containerRef = useRef<HTMLDivElement>(null);
  const [scale, setScale] = useState(1);
  const config = DEVICE_CONFIGS[device];

  useEffect(() => {
    if (device === 'desktop') {
      setScale(1);
      return;
    }

    const updateScale = () => {
      if (!containerRef.current) return;

      const container = containerRef.current;
      const availableWidth = container.clientWidth - 48; // padding
      const availableHeight = window.innerHeight - 120; // context bar + padding

      const scaleX = availableWidth / config.width;
      const scaleY = availableHeight / config.height;
      const newScale = Math.min(scaleX, scaleY, 1); // Don't scale up

      setScale(newScale);
    };

    updateScale();
    window.addEventListener('resize', updateScale);
    return () => window.removeEventListener('resize', updateScale);
  }, [device, config.width, config.height]);

  if (device === 'desktop') {
    return <div className="device-frame-desktop">{children}</div>;
  }

  return (
    <div ref={containerRef} className="device-frame-container">
      <style jsx>{`
        .device-frame-container {
          display: flex;
          flex-direction: column;
          align-items: center;
          justify-content: flex-start;
          padding: 24px;
          min-height: calc(100vh - 60px);
          background: var(--page-bg);
        }

        .device-frame-desktop {
          min-height: calc(100vh - 60px);
        }

        .device-label {
          font-size: 12px;
          color: var(--text-tertiary);
          margin-bottom: 12px;
          font-family: system-ui, sans-serif;
        }

        .device-bezel {
          background: #1a1a1a;
          border-radius: ${device === 'iphone' ? '40px' : '20px'};
          padding: ${device === 'iphone' ? '12px' : '16px'};
          box-shadow: 
            0 0 0 1px rgba(255, 255, 255, 0.1),
            0 25px 50px -12px rgba(0, 0, 0, 0.5);
        }

        .device-screen {
          width: ${config.width}px;
          height: ${config.height}px;
          background: var(--page-bg);
          border-radius: ${device === 'iphone' ? '32px' : '8px'};
          overflow: hidden;
          transform: scale(${scale});
          transform-origin: top left;
        }

        .device-bezel-wrapper {
          width: ${config.width * scale + (device === 'iphone' ? 24 : 32)}px;
          height: ${config.height * scale + (device === 'iphone' ? 24 : 32)}px;
        }

        .device-notch {
          position: absolute;
          top: ${device === 'iphone' ? '12px' : '0'};
          left: 50%;
          transform: translateX(-50%);
          width: ${device === 'iphone' ? '120px' : '0'};
          height: ${device === 'iphone' ? '28px' : '0'};
          background: #1a1a1a;
          border-radius: 0 0 20px 20px;
          z-index: 10;
        }

        .device-home-indicator {
          position: absolute;
          bottom: ${device === 'iphone' ? '8px' : '0'};
          left: 50%;
          transform: translateX(-50%);
          width: ${device === 'iphone' ? '120px' : '0'};
          height: ${device === 'iphone' ? '4px' : '0'};
          background: rgba(255, 255, 255, 0.3);
          border-radius: 2px;
        }

        .screen-content {
          width: 100%;
          height: 100%;
          overflow: auto;
        }
      `}</style>

      <div className="device-label">{config.name} Preview</div>
      <div className="device-bezel-wrapper">
        <div className="device-bezel" style={{ position: 'relative' }}>
          {device === 'iphone' && <div className="device-notch" />}
          <div className="device-screen">
            <div className="screen-content">{children}</div>
          </div>
          {device === 'iphone' && <div className="device-home-indicator" />}
        </div>
      </div>
    </div>
  );
}
