'use client';

import { useState } from 'react';

export type Theme = 'light' | 'dark' | 'wireframe';
export type DeviceType = 'desktop' | 'ipad' | 'iphone';

interface ContextBarProps {
  title: string;
  who: string;
  what: string;
  why: string;
  currentTheme: Theme;
  currentDevice: DeviceType;
  onThemeChange: (theme: Theme) => void;
  onDeviceChange: (device: DeviceType) => void;
}

export function ContextBar({
  title,
  who,
  what,
  why,
  currentTheme,
  currentDevice,
  onThemeChange,
  onDeviceChange,
}: ContextBarProps) {
  const [isExpanded, setIsExpanded] = useState(false);

  return (
    <div className="context-bar">
      <style jsx>{`
        .context-bar {
          position: sticky;
          top: 0;
          z-index: 1000;
          background: var(--cb-bg, #0a0a0a);
          border-bottom: 1px solid var(--cb-border, #2a2a2a);
          font-family: system-ui, -apple-system, sans-serif;
        }

        .context-bar-main {
          display: flex;
          align-items: center;
          justify-content: space-between;
          padding: 12px 16px;
          gap: 16px;
        }

        .context-bar-left {
          display: flex;
          align-items: center;
          gap: 12px;
          min-width: 0;
        }

        .context-bar-title {
          font-size: 14px;
          font-weight: 600;
          color: var(--cb-text, #f5f5f5);
          white-space: nowrap;
          overflow: hidden;
          text-overflow: ellipsis;
        }

        .context-bar-center {
          flex: 1;
          display: flex;
          justify-content: center;
        }

        .context-toggle {
          display: flex;
          align-items: center;
          gap: 6px;
          padding: 6px 12px;
          background: var(--cb-button-bg, #2a2a2a);
          border: none;
          border-radius: 6px;
          color: var(--cb-text-secondary, #a3a3a3);
          font-size: 12px;
          cursor: pointer;
          transition: background-color 0.15s;
        }

        .context-toggle:hover {
          background: var(--cb-button-bg-hover, #404040);
        }

        .context-toggle-icon {
          transition: transform 0.2s;
        }

        .context-toggle-icon.expanded {
          transform: rotate(180deg);
        }

        .context-bar-right {
          display: flex;
          align-items: center;
          gap: 8px;
        }

        .theme-buttons, .device-buttons {
          display: flex;
          background: var(--cb-button-bg, #2a2a2a);
          border-radius: 6px;
          padding: 2px;
        }

        .theme-btn, .device-btn {
          padding: 6px 10px;
          background: transparent;
          border: none;
          border-radius: 4px;
          color: var(--cb-text-secondary, #a3a3a3);
          font-size: 11px;
          cursor: pointer;
          transition: all 0.15s;
        }

        .theme-btn:hover, .device-btn:hover {
          color: var(--cb-text, #f5f5f5);
        }

        .theme-btn.active, .device-btn.active {
          background: var(--cb-button-bg-active, #525252);
          color: var(--cb-text, #f5f5f5);
        }

        .context-panel {
          display: none;
          padding: 0 16px 16px;
          gap: 24px;
        }

        .context-panel.expanded {
          display: grid;
          grid-template-columns: repeat(3, 1fr);
        }

        .context-section {
          padding: 12px;
          background: var(--cb-button-bg, #2a2a2a);
          border-radius: 8px;
        }

        .context-section-label {
          font-size: 10px;
          font-weight: 600;
          text-transform: uppercase;
          letter-spacing: 0.5px;
          color: var(--cb-text-secondary, #a3a3a3);
          margin-bottom: 6px;
        }

        .context-section-content {
          font-size: 13px;
          color: var(--cb-text, #f5f5f5);
          line-height: 1.4;
        }

        @media (max-width: 768px) {
          .context-panel.expanded {
            grid-template-columns: 1fr;
          }

          .context-bar-center {
            display: none;
          }

          .theme-buttons, .device-buttons {
            display: none;
          }
        }
      `}</style>

      <div className="context-bar-main">
        <div className="context-bar-left">
          <span className="context-bar-title">{title}</span>
        </div>

        <div className="context-bar-center">
          <button
            className="context-toggle"
            onClick={() => setIsExpanded(!isExpanded)}
          >
            <span>Context</span>
            <svg
              className={`context-toggle-icon ${isExpanded ? 'expanded' : ''}`}
              width="12"
              height="12"
              viewBox="0 0 12 12"
              fill="none"
            >
              <path
                d="M3 4.5L6 7.5L9 4.5"
                stroke="currentColor"
                strokeWidth="1.5"
                strokeLinecap="round"
                strokeLinejoin="round"
              />
            </svg>
          </button>
        </div>

        <div className="context-bar-right">
          <div className="theme-buttons">
            <button
              className={`theme-btn ${currentTheme === 'light' ? 'active' : ''}`}
              onClick={() => onThemeChange('light')}
              title="Light mode"
            >
              Light
            </button>
            <button
              className={`theme-btn ${currentTheme === 'dark' ? 'active' : ''}`}
              onClick={() => onThemeChange('dark')}
              title="Dark mode"
            >
              Dark
            </button>
            <button
              className={`theme-btn ${currentTheme === 'wireframe' ? 'active' : ''}`}
              onClick={() => onThemeChange('wireframe')}
              title="Wireframe mode"
            >
              Wire
            </button>
          </div>

          <div className="device-buttons">
            <button
              className={`device-btn ${currentDevice === 'desktop' ? 'active' : ''}`}
              onClick={() => onDeviceChange('desktop')}
              title="Desktop view"
            >
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2">
                <rect x="2" y="3" width="20" height="14" rx="2" />
                <line x1="8" y1="21" x2="16" y2="21" />
                <line x1="12" y1="17" x2="12" y2="21" />
              </svg>
            </button>
            <button
              className={`device-btn ${currentDevice === 'ipad' ? 'active' : ''}`}
              onClick={() => onDeviceChange('ipad')}
              title="iPad view"
            >
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2">
                <rect x="4" y="2" width="16" height="20" rx="2" />
                <circle cx="12" cy="18" r="1" />
              </svg>
            </button>
            <button
              className={`device-btn ${currentDevice === 'iphone' ? 'active' : ''}`}
              onClick={() => onDeviceChange('iphone')}
              title="iPhone view"
            >
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2">
                <rect x="6" y="2" width="12" height="20" rx="2" />
                <line x1="10" y1="5" x2="14" y2="5" />
              </svg>
            </button>
          </div>
        </div>
      </div>

      <div className={`context-panel ${isExpanded ? 'expanded' : ''}`}>
        <div className="context-section">
          <div className="context-section-label">Who is this for</div>
          <div className="context-section-content">{who}</div>
        </div>
        <div className="context-section">
          <div className="context-section-label">What you can do</div>
          <div className="context-section-content">{what}</div>
        </div>
        <div className="context-section">
          <div className="context-section-label">Why this matters</div>
          <div className="context-section-content">{why}</div>
        </div>
      </div>
    </div>
  );
}
