/**
 * Parse CLAUDE.md files to extract prototype metadata
 * 
 * Expected format:
 * # Prototype Name
 * 
 * ## Who is this for
 * Target audience description...
 * 
 * ## What you can do
 * Key functionality...
 * 
 * ## Why this matters
 * Value proposition...
 */

export interface PrototypeContext {
  title: string;
  who: string;
  what: string;
  why: string;
  raw: string;
}

const DEFAULT_CONTEXT: PrototypeContext = {
  title: 'Untitled Prototype',
  who: 'No audience defined yet',
  what: 'No functionality described yet',
  why: 'No value proposition defined yet',
  raw: '',
};

/**
 * Extract a section from markdown content
 * Looks for ## heading and captures content until next ## or end
 */
function extractSection(content: string, heading: string): string {
  // Case-insensitive regex to find the section
  const regex = new RegExp(
    `##\\s*${heading}[\\s\\S]*?\\n([\\s\\S]*?)(?=\\n##|$)`,
    'i'
  );
  
  const match = content.match(regex);
  if (!match || !match[1]) return '';
  
  return match[1].trim();
}

/**
 * Extract the main title (# heading) from markdown
 */
function extractTitle(content: string): string {
  const match = content.match(/^#\s+(.+)$/m);
  return match ? match[1].trim() : '';
}

/**
 * Parse CLAUDE.md content and extract structured metadata
 */
export function parseClaudeMd(content: string): PrototypeContext {
  if (!content || typeof content !== 'string') {
    return { ...DEFAULT_CONTEXT };
  }

  const title = extractTitle(content) || DEFAULT_CONTEXT.title;
  
  // Try various heading variations
  const who = 
    extractSection(content, 'Who is this for') ||
    extractSection(content, 'Who') ||
    extractSection(content, 'Audience') ||
    extractSection(content, 'Target audience') ||
    DEFAULT_CONTEXT.who;

  const what = 
    extractSection(content, 'What you can do') ||
    extractSection(content, 'What') ||
    extractSection(content, 'Features') ||
    extractSection(content, 'Functionality') ||
    DEFAULT_CONTEXT.what;

  const why = 
    extractSection(content, 'Why this matters') ||
    extractSection(content, 'Why') ||
    extractSection(content, 'Value') ||
    extractSection(content, 'Purpose') ||
    DEFAULT_CONTEXT.why;

  return {
    title,
    who,
    what,
    why,
    raw: content,
  };
}

/**
 * Load and parse CLAUDE.md from file system (server-side only)
 * Falls back to default context if file doesn't exist
 */
export async function loadClaudeMd(basePath: string = process.cwd()): Promise<PrototypeContext> {
  try {
    // Dynamic import for server-side only
    const fs = await import('fs/promises');
    const path = await import('path');
    
    const filePath = path.join(basePath, 'CLAUDE.md');
    const content = await fs.readFile(filePath, 'utf-8');
    
    return parseClaudeMd(content);
  } catch {
    // File doesn't exist or can't be read
    return { ...DEFAULT_CONTEXT };
  }
}

export { DEFAULT_CONTEXT };
