# Wireframe Kit

A self-contained environment for generating interactive prototypes with theme support and device previews — designed to work with AI tools like Claude, Cursor, and ChatGPT.

---

## What This Is

A prototyping environment with:
- **Three theme modes**: Light, Dark, and Wireframe (grayscale)
- **Device previews**: Desktop, iPad, and iPhone frames with scaling
- **Context bar**: Shows who the prototype is for, what it does, and why it matters
- **CLAUDE.md integration**: Automatically displays context from your documentation

Use it to quickly mock up UX ideas and share them with stakeholders.

---

## Quick Start

```bash
# From the wireframe-kit directory
npm install
npm run dev
```

Then open `http://localhost:3000` — you'll see the context bar at the top with theme and device toggles.

---

## CLAUDE.md — Prototype Context

Create a `CLAUDE.md` file in the root of your wireframe-kit to define context that appears in the top bar:

```markdown
# My Prototype Name

## Who is this for
Describe the target audience...

## What you can do
Key functionality and interactions...

## Why this matters
The value proposition or problem being solved...

## Build History
- v0.1: Initial concept
- v0.2: Added filtering
```

The context bar will automatically parse and display the Who/What/Why sections.

---

## Theme System

Three themes available via the context bar:

| Theme | Description |
|-------|-------------|
| **Light** | White backgrounds, dark text — professional look |
| **Dark** | Dark backgrounds, light text — modern feel |
| **Wireframe** | Pure grayscale, dashed borders — sketch aesthetic |

Themes are saved to localStorage and persist across sessions.

### CSS Variables

Use these variables for theme-aware styling:

```css
/* Surfaces */
var(--surface-bg)        /* Main background */
var(--surface-secondary) /* Secondary surfaces */
var(--surface-elevated)  /* Cards, elevated elements */

/* Text */
var(--text-primary)      /* Main text */
var(--text-secondary)    /* Secondary text */
var(--text-tertiary)     /* Muted text */
var(--text-muted)        /* Very subtle text */

/* Borders */
var(--border-default)    /* Standard borders */
var(--border-strong)     /* Emphasized borders */

/* Interactive */
var(--interactive-bg)    /* Primary button background */
var(--interactive-text)  /* Primary button text */

/* Page */
var(--page-bg)           /* Page background */

/* Shadows */
var(--shadow-sm)         /* Subtle shadow */
var(--shadow-md)         /* Medium shadow */
var(--shadow-lg)         /* Large shadow */
```

---

## Device Preview

The context bar includes device toggles:

| Device | Dimensions | Use Case |
|--------|------------|----------|
| Desktop | Full width | Standard view |
| iPad | 1024 × 768 | Tablet testing |
| iPhone | 390 × 844 | Mobile testing |

The prototype is wrapped in a device frame and scaled to fit the viewport.

---

## Available Primitives

| Component | Usage |
|-----------|-------|
| `Button` | `<Button variant="primary">Label</Button>` |
| `Input` | `<Input label="Name" placeholder="..." />` |
| `Card` | `<Card variant="elevated">content</Card>` |
| `Divider` | `<Divider spacing="lg" />` |

### Button Variants
- `default` — Secondary style with border
- `primary` — Filled, high emphasis
- `ghost` — Transparent, minimal

### Card Variants
- `default` — Standard card
- `elevated` — With shadow
- `outlined` — Thicker border, no shadow

Import from:
```tsx
import { Card } from '../wireframe-primitives/Card';
import { Button } from '../wireframe-primitives/Button';
import { Input } from '../wireframe-primitives/Input';
import { Divider } from '../wireframe-primitives/Divider';
```

---

## File Structure

```
wireframe-kit/
├── app/
│   ├── layout.tsx              # Root layout with PrototypeShell
│   ├── page.tsx                # Landing page
│   └── experiments/            # Your prototypes go here
├── lib/
│   └── parseClaudeMd.ts        # CLAUDE.md parser
├── styles/
│   ├── themes.css              # Light/Dark/Wireframe themes
│   └── wireframe.css           # Base styles
├── wireframe-primitives/
│   ├── Button.tsx
│   ├── Card.tsx
│   ├── Input.tsx
│   ├── Divider.tsx
│   ├── ContextBar.tsx          # Top bar with context
│   ├── DeviceFrame.tsx         # Device preview wrapper
│   └── PrototypeShell.tsx      # Combines ContextBar + DeviceFrame
├── CLAUDE.md                   # Your prototype's context (create this!)
├── CLAUDE.md.template          # Template to copy
└── README-for-AI.md            # This file
```

---

## Example Prompts for AI Tools

### 1. Create a new prototype

> Create a new experiment in `wireframe-kit/app/experiments/dashboard/page.tsx`.
> 
> Build a simple dashboard with:
> - A header showing "Welcome back, [User]"
> - 3 stat cards showing KPIs
> - A recent activity list
> 
> Use the Card, Button, Divider primitives.
> Make sure it looks good in all three themes.

### 2. Add a form

> In my dashboard experiment, add a settings section with:
> - Name input field
> - Email input field  
> - Notification toggle (use a checkbox)
> - Save button (primary variant)
> 
> Use the Input component with labels.

### 3. Create CLAUDE.md for my prototype

> Create a CLAUDE.md file for my dashboard prototype with:
> - Title: Analytics Dashboard
> - Who: Product managers tracking team metrics
> - What: View KPIs, recent activity, and trends
> - Why: Quick visibility into team performance without digging through reports

### 4. Test mobile responsiveness

> Review my dashboard experiment and ensure it works well on iPhone.
> - Stack cards vertically on small screens
> - Make the header more compact
> - Ensure touch targets are at least 44px

---

## Rules for Extending

1. **Self-contained** — No imports from outside `wireframe-kit/`. Use relative paths.

2. **Theme-aware** — Use CSS variables, not hardcoded colors.

3. **Local mock data** — Define constants in the same file:
   ```tsx
   const ITEMS = [
     { id: "1", name: "...", status: "pending" },
   ];
   ```

4. **Simple state** — Use `useState` for interactivity. No complex state management.

5. **Single file preferred** — Keep experiments in one `page.tsx` unless components are reusable.

6. **Update CLAUDE.md** — Keep the context file updated as you build.

---

## Deploying to VibeSharing

When your prototype is ready to share:

1. Ensure `CLAUDE.md` is up to date with who/what/why
2. Zip the wireframe-kit folder (or push to GitHub)
3. Deploy to VibeSharing — the CLAUDE.md will auto-import as context

The context bar and device preview are for local development. When deployed, VibeSharing provides its own navigation.
